<?php
/**
 * 访客IP数据提供者
 * 处理独立访客IP统计的数据获取逻辑
 */

require_once __DIR__ . '/DataProviderInterface.php';
require_once __DIR__ . '/FilterCriteria.php';

class IpDataProvider implements DataProviderInterface {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * 获取独立IP总数
     */
    public function getTotalCount(FilterCriteria $filter) {
        $whereData = $filter->buildWhereConditions();
        
        $sql = "SELECT COUNT(DISTINCT ip) as total FROM pageviews " . $whereData['where'];
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($whereData['params']);
        
        return (int)$stmt->fetch()['total'];
    }
    
    /**
     * 获取IP分页数据
     */
    public function getPageData(FilterCriteria $filter, $offset, $limit, $sortBy = null) {
        $whereData = $filter->buildWhereConditions();
        
        // 先获取总访问数用于计算百分比
        $totalSql = "SELECT COUNT(*) as total FROM pageviews " . $whereData['where'];
        $totalStmt = $this->pdo->prepare($totalSql);
        $totalStmt->execute($whereData['params']);
        $totalVisits = $totalStmt->fetch()['total'];
        
        // 获取分页数据
        $sql = "
            SELECT 
                ip,
                COUNT(*) as visits,
                ROUND(COUNT(*) * 100.0 / ?, 2) as percentage,
                MAX(visit_time) as last_visit,
                SUBSTRING_INDEX(GROUP_CONCAT(
                    CASE 
                        WHEN referer IS NULL OR referer = '' THEN '直接访问'
                        WHEN referer LIKE '%google%' THEN 'Google'
                        WHEN referer LIKE '%baidu%' THEN '百度'
                        WHEN referer LIKE '%bing%' THEN 'Bing'
                        WHEN referer LIKE '%yahoo%' THEN 'Yahoo'
                        WHEN referer LIKE '%sogou%' THEN '搜狗'
                        WHEN referer LIKE '%so.com%' OR referer LIKE '%360.cn%' THEN '360'
                        ELSE '外链'
                    END 
                    ORDER BY visit_time DESC
                ), ',', 1) as main_source
            FROM pageviews 
            " . $whereData['where'] . "
            GROUP BY ip
            ORDER BY visits DESC, last_visit DESC
            LIMIT ? OFFSET ?
        ";
        
        $params = array_merge([$totalVisits], $whereData['params'], [$limit, $offset]);
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>

