<?php
/**
 * 数据库连接和表结构检查脚本
 * 在注入数据前先验证数据库状态
 */

echo "🔍 数据库连接和表结构检查\n";
echo "================================\n\n";

// 数据库配置
$dbConfig = [
    'host' => 'localhost',
    'port' => '3306', 
    'dbname' => 'jef_analytics_mvp',
    'user' => 'root',
    'pass' => 'zoujifu123'
];

try {
    echo "📡 尝试连接数据库...\n";
    echo "主机: {$dbConfig['host']}:{$dbConfig['port']}\n";
    echo "数据库: {$dbConfig['dbname']}\n";
    echo "用户: {$dbConfig['user']}\n\n";
    
    $dsn = "mysql:host={$dbConfig['host']};port={$dbConfig['port']};dbname={$dbConfig['dbname']};charset=utf8mb4";
    $pdo = new PDO($dsn, $dbConfig['user'], $dbConfig['pass'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false
    ]);
    
    echo "✅ 数据库连接成功！\n\n";
    
    // 检查数据库版本
    $version = $pdo->query("SELECT VERSION() as version")->fetch();
    echo "📊 MySQL版本: {$version['version']}\n\n";
    
    // 检查pageviews表是否存在
    echo "🔍 检查pageviews表...\n";
    $tables = $pdo->query("SHOW TABLES LIKE 'pageviews'")->fetchAll();
    
    if (empty($tables)) {
        echo "❌ pageviews表不存在！\n";
        echo "💡 请先运行数据库初始化脚本：\n";
        echo "   mysql -u root -p {$dbConfig['dbname']} < database/init.sql\n\n";
        exit(1);
    }
    
    echo "✅ pageviews表存在\n\n";
    
    // 检查表结构
    echo "📋 检查pageviews表结构...\n";
    $columns = $pdo->query("DESCRIBE pageviews")->fetchAll();
    
    echo "字段列表:\n";
    $requiredFields = ['id', 'site_key', 'ip', 'user_agent', 'page_url', 'referer', 'visit_time', 'is_bot', 'bot_score'];
    $existingFields = [];
    
    foreach ($columns as $column) {
        $field = $column['Field'];
        $type = $column['Type'];
        $null = $column['Null'];
        $default = $column['Default'];
        
        echo "  - {$field}: {$type}";
        if ($null === 'NO') echo " NOT NULL";
        if ($default !== null) echo " DEFAULT '{$default}'";
        echo "\n";
        
        $existingFields[] = $field;
    }
    
    echo "\n";
    
    // 检查必需字段
    $missingFields = array_diff($requiredFields, $existingFields);
    if (!empty($missingFields)) {
        echo "❌ 缺少必需字段: " . implode(', ', $missingFields) . "\n";
        exit(1);
    }
    
    echo "✅ 所有必需字段都存在\n\n";
    
    // 检查现有数据
    echo "📊 检查现有数据...\n";
    $stats = $pdo->query("
        SELECT 
            COUNT(*) as total_records,
            COUNT(CASE WHEN is_bot = 1 THEN 1 END) as bot_records,
            COUNT(CASE WHEN is_bot = 0 THEN 1 END) as human_records,
            MIN(visit_time) as earliest_record,
            MAX(visit_time) as latest_record
        FROM pageviews
    ")->fetch();
    
    echo "总记录数: {$stats['total_records']}\n";
    echo "机器人记录: {$stats['bot_records']}\n";
    echo "真实用户记录: {$stats['human_records']}\n";
    
    if ($stats['total_records'] > 0) {
        echo "最早记录: {$stats['earliest_record']}\n";
        echo "最新记录: {$stats['latest_record']}\n";
    }
    
    echo "\n";
    
    // 检查sites表
    echo "🔍 检查sites表...\n";
    $sitesTable = $pdo->query("SHOW TABLES LIKE 'sites'")->fetchAll();
    
    if (empty($sitesTable)) {
        echo "❌ sites表不存在！\n";
        exit(1);
    }
    
    echo "✅ sites表存在\n";
    
    $sites = $pdo->query("SELECT * FROM sites")->fetchAll();
    echo "站点数量: " . count($sites) . "\n";
    
    // 检查默认站点
    $defaultSite = null;
    foreach ($sites as $site) {
        echo "  - {$site['name']} (key: {$site['site_key']}, domain: {$site['domain']}, 状态: {$site['status']})\n";
        if ($site['site_key'] === 'default') {
            $defaultSite = $site;
        }
    }
    
    if (!$defaultSite) {
        echo "⚠️ 缺少 'default' 站点！这是系统必需的默认站点\n";
        echo "💡 请在 sites 表中添加默认站点或重新运行 init.sql\n\n";
    } else {
        echo "✅ 默认站点存在\n";
    }
    
    echo "\n";
    
    // 检查user_preferences表（新增）
    echo "🔍 检查user_preferences表...\n";
    $userPrefTable = $pdo->query("SHOW TABLES LIKE 'user_preferences'")->fetchAll();
    
    if (empty($userPrefTable)) {
        echo "⚠️ user_preferences表不存在！\n";
        echo "💡 这是新功能所需的表，用于保存用户的默认站点选择\n";
        echo "💡 请执行以下SQL创建该表：\n\n";
        echo "CREATE TABLE IF NOT EXISTS `user_preferences` (\n";
        echo "    `username` VARCHAR(191) PRIMARY KEY,\n";
        echo "    `default_site_id` INT NULL,\n";
        echo "    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\n";
        echo "    INDEX `idx_site_id` (`default_site_id`)\n";
        echo ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;\n\n";
    } else {
        echo "✅ user_preferences表存在\n";
        
        // 检查表结构
        $prefColumns = $pdo->query("DESCRIBE user_preferences")->fetchAll();
        $prefFields = array_column($prefColumns, 'Field');
        $requiredPrefFields = ['username', 'default_site_id', 'updated_at'];
        $missingPrefFields = array_diff($requiredPrefFields, $prefFields);
        
        if (!empty($missingPrefFields)) {
            echo "⚠️ user_preferences表缺少字段: " . implode(', ', $missingPrefFields) . "\n";
        } else {
            echo "✅ user_preferences表结构完整\n";
        }
        
        // 显示用户偏好统计
        $prefCount = $pdo->query("SELECT COUNT(*) as count FROM user_preferences")->fetch();
        echo "已保存偏好的用户数: {$prefCount['count']}\n";
    }
    
    echo "\n";
    
    // 检查索引
    echo "🔍 检查pageviews表索引...\n";
    $indexes = $pdo->query("SHOW INDEX FROM pageviews")->fetchAll();
    
    $indexNames = [];
    foreach ($indexes as $index) {
        $indexNames[] = $index['Key_name'];
    }
    $indexNames = array_unique($indexNames);
    
    echo "索引列表: " . implode(', ', $indexNames) . "\n\n";
    
    // 检查重要索引
    $requiredIndexes = ['idx_is_bot', 'idx_site_key', 'idx_visit_time'];
    $missingIndexes = array_diff($requiredIndexes, $indexNames);
    
    if (!empty($missingIndexes)) {
        echo "⚠️ 缺少推荐索引: " . implode(', ', $missingIndexes) . "\n";
        echo "💡 这可能影响查询性能，但不影响功能\n\n";
    } else {
        echo "✅ 重要索引都存在\n\n";
    }
    
    echo "🎉 数据库检查完成！\n";
    echo "✅ 数据库状态正常，可以安全注入测试数据\n\n";
    
    // 返回连接对象供后续使用
    return $pdo;
    
} catch (PDOException $e) {
    echo "❌ 数据库连接失败！\n";
    echo "错误信息: " . $e->getMessage() . "\n";
    echo "错误代码: " . $e->getCode() . "\n\n";
    
    echo "💡 可能的解决方案:\n";
    echo "1. 检查MySQL服务是否启动\n";
    echo "2. 验证数据库配置信息\n";
    echo "3. 确认数据库 '{$dbConfig['dbname']}' 已创建\n";
    echo "4. 检查用户权限\n\n";
    
    exit(1);
} catch (Exception $e) {
    echo "❌ 检查过程中发生错误！\n";
    echo "错误信息: " . $e->getMessage() . "\n\n";
    exit(1);
}
?>
